
/* Copyright (C) 2001-2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_fixed.c */


/****************************************************************/
/*********** IMPORTANT PERFORMANCE NOTE *************************/
/****************************************************************/
/*
 * This is a good place to improve performance.  These fixed
 * point routines are used very often in all aspects of the
 * program. Make sure you're using the best code for your box.
 *
 * In order of speed the program prefers to use:
 * 1) native 64 bit integer code                   XXXXXXX_64
 * 2) hand coded assembly routines                 XXXXXX_ASM
 * 3) compiler supported 64 bit integer code       XXXXXXX_64
 * 4) 32 bit ANSI C code                           XXXXXXX
 *
 * If your chip does not have native 64 bit integers, but does
 * have 32x32=64 multiply and 64/32=32 divide instructions (like
 * the intel 80x86 or motorola 680x0 chips) it is probably VERY
 * worthwhile to write the XXXXXX_ASM variants to take advantage
 * of these instructions.  For example, using the MSVC compiler
 * on a Pentium we have the following timings:
 *
 *                 ANSI          _64     _ASM
 * muldiv        22630        11970     8680
 * varmul         8070         6490     5710
 * vardiv        11920        10320     6810
 *
 * If you write the XXXXXX_ASM variants, be SURE that they agree
 * with the ANSI C versions -- run VALUE_TEST to be sure.
 *
 */
/****************************************************************/
/****************************************************************/
/****************************************************************/


#include "fs_object.h"
#include "fs_fixed.h"

/* set either or both to (1) for standalone test program */
#define TIMING_TEST (0)
#define VALUE_TEST (0)
#define EITHER_TEST (TIMING_TEST || VALUE_TEST)

/****************************************************************/
/* Copyright 2000 Apple Computer, all rights reserved */
/* square root of a FRACT (2.30 fixed point) */
FRACT FracSqrt (FRACT xf)
{
    FRACT b = 0L;
    FS_ULONG c, d, x = xf;

    if (xf < 0)
        return -MYINFINITY;
    if (xf == 0)
        return 0;

    if (x >= 0x40000000L)
    {
        x -= 0x40000000L;
        b  = 0x40000000L;
    }

    for (c = 0x10000000L; c; c >>= 1)
    {
        d = b + c;
        if (x >= d)
        {
            x -= d;
            b += (c << 1);
        }
        x <<= 1;
    }

    if (x > (FS_ULONG)b)
    {
        x -= b;
        x <<= 1;
        x--;
        b++;
    }
    else
    {
        x <<= 1;
    }

    return ( b + (FRACT)(x > (FS_ULONG)b) );
}

/****************************************************************/
/* factor out 4^k, use the 2.30 sqrt above, shift back to 16.16 */
FS_FIXED FixSqrt(FS_FIXED x)
{
    FS_ULONG r, f;
    int k = 7;

    if (x < 0)
        return -MYINFINITY;
    if (x == 0)
        return 0;

    f = x;
    while (f < 0x20000000)
    {
        f <<= 2;
        k++;
    }

    r = FracSqrt(f);
    return (r + (1 << (k - 1))) >> k;
}

/****************************************************************/
/* ROUGH approximation to sqrt(dx*dx+dy*dy) */
FS_FIXED DistanceNorm(FS_FIXED dx, FS_FIXED dy)
{
    if (dx < 0) dx = -dx;
    if (dy < 0) dy = -dy;
    return (dx > dy) ? dx + dy / 3 : dy + dx / 3;
}

/****************************************************************/
/* make <p> into a unit vector ... ie: make the square root
 * of the sums of the squares of p->x and p->y == FIXED_ONE
 * This is correct to within 1/65536-th of a pixel
 */
FS_VOID fixed_norm(FIXED_VECTOR *p)
{
    FS_ULONG x, b, c, d;
    FS_FIXED dx, dy, max = 0;

    dx = p->x;
    dy = p->y;

    if (dy == 0)
    {
        p->x = (p->x < 0) ? -65536 : 65536;
        return;
    }
    if (dx == 0)
    {
        p->y = (p->y < 0) ? -65536 : 65536;
        return;
    }

    /* quick pre-normalization ... to get ABS(dx) and ABS(dy) <= FIXED_ONE */
    {
        FS_FIXED adx = dx, ady = dy;
        if (adx < 0) adx = -adx;
        if (ady < 0) ady = -ady;
        if (adx > ady) max = adx;
        else           max = ady;
    }

    if (max > FIXED_ONE)
    {
        dx = FixDiv(dx, max);
        dy = FixDiv(dy, max);
    }
    else if (max < 256) /* max < sqrt(1/65536) */
    {
        /* scale tiny vectors up so that x below is not zero */
        dx = LongMulDiv(dx, 256, max);
        dy = LongMulDiv(dy, 256, max);
    }


    /* sum of the squares -- now known to be in the inverval [zero,FIXED_TWO] */
    x = FixMul(dx, dx) + FixMul(dy, dy);

    /* extract the square root <b> of <x> -- from some APPLE code
    * assumes 16.16 fixed point number in interval (0,FIXED_TWO]
    * and it's reasonably fast ... only 16 iterations */
    for (b = 0, c = 0x00010000; c; c >>= 1)
    {
        d = b + c;
        if (x >= d)
        {
            x -= d;
            b += (c << 1);
        }
        x <<= 1;
    }
    b >>= 1;

    /* divide each coordinate by square root of the sum of the squares */
    p->x = FixDiv(dx, b);
    p->y = FixDiv(dy, b);
}

/****************************************************************/
FS_FIXED vardivNoRound(FS_FIXED x, FS_FIXED y, int n)
{
    int neg;
    FS_FIXED q;
    FS_ULONG w, d;   /* q = w/d */
    /* FS_USHORT qct; */
    FS_SHORT qct;    /* 02-19-01 */

    if (y == 0)
        return (x >= 0) ? MYINFINITY : -MYINFINITY;
    if (x == 0)
        return 0L;

    neg = (x ^ y) < 0;
    if (x < 0) x = -x;
    if (y < 0) y = -y;

    w = x;
    d = y;
    qct = (FS_SHORT)(n + 1);  /* ABS(n)<32 */
    while (w < d)
    {
        w <<= 1;
        qct--;
    }
    while (d <= w)
    {
        d <<= 1;
        qct++;
    }
    d >>= 1;
    qct--;

    /* 02-19-01 */
    if (qct < 0)    /* <w> was too small with respect to <d> ... result is 0 */
        return 0;
    /* 02-19-01 */

    if (qct > 31)
        return neg ? -MYINFINITY : MYINFINITY;

    q = 0;
    while (qct)
    {
        q <<= 1;
        if (d <= w)      /* w==x, d==y */
        {
            w -= d;
            q |= 1;
        }
        w <<= 1;
        qct--;
    }

    return neg ? -q : q;
}

/****************************************************************/
/**************** 32 bit ANSI C *********************************/
/****************************************************************/

#if !defined(HAS_ASM) && !defined(HAS_FS_INT64)
FS_LONG muldiv(FS_LONG a, FS_LONG b, FS_LONG c)
{
    FS_ULONG lo, m1, m2, hi, q;
    int i, neg;

    /* check for 0's */
    if (c == 0)
        return ((a ^ b) < 0) ? -MYINFINITY : MYINFINITY;
    if (a == 0 || b == 0)
        return 0;

    /* make all args positive, keeping sign of result */
    neg = (a ^ b ^ c) < 0;
    if (a < 0) a = -a;
    if (b < 0) b = -b;
    if (c < 0) c = -c;

    /* 4 partial products */
    lo = LO_WORD(a) * LO_WORD(b);   /*         [  lo   ]    */
    m1 = LO_WORD(a) * HI_WORD(b);   /*     [  m1  ]         */
    m2 = HI_WORD(a) * LO_WORD(b);   /*     [  m2  ]         */
    hi = HI_WORD(a) * HI_WORD(b);   /* [  hi  ]             */

    /* add m1 and m2 (middles) and carry into HI_WORD(hi) as needed */
    if (m2 && m1 >= (1 + ~m2))
        hi += (1L << 16);
    m1 += m2;

    /* seperate halves of (middles) */
    m2 = m1 >> 16;        /* HI_WORD(middles) aligned with hi */
    m1 = m1 << 16;        /* LO_WORD(middles) aligned with lo */

    /* add HI_WORD(middles) to hi */
    hi += m2;

    /* add LO_WORD(middles) to lo ... carry into hi as needed */
    if (m1 && lo >= (1 + ~m1))
        hi++;
    lo += m1;

    /* to effect rounding, add c/2 to lo ... carry into hi as needed */
    m1 = (c >> 1);
    if (m1 && lo >= (1 + ~m1))
        hi++;
    lo += m1;

    /* will divide overflow ? */
    if (hi >= (FS_ULONG)c)
        return ( neg ) ? -MYINFINITY : MYINFINITY;

    /* schoolboy long division in base 2 */
    q = 0;
    for ( i = 0; i < 32; i++ )
    {
        hi <<= 1;
        hi  |= lo >> 31;
        q <<= 1;

        if ( hi >= (FS_ULONG)c )
        {
            hi -= c;
            q |= 1;
        }
        lo <<= 1;
    }

    /* overflow */
    if (q > MYINFINITY)
        return neg ? -MYINFINITY : MYINFINITY;

    return neg ? -(FS_LONG)q : (FS_LONG)q;
}
/****************************************************************/
FS_LONG varmul(FS_LONG a, FS_LONG b, int n)
{
    FS_ULONG lo, m1, m2, hi;
    int neg;

    if (a == 0 || b == 0)
        return 0;

    neg = (a ^ b) < 0;
    if (a < 0) a = -a;
    if (b < 0) b = -b;

    /* 4 partial products */
    lo = LO_WORD(a) * LO_WORD(b);   /*         [  lo   ]    */
    m1 = LO_WORD(a) * HI_WORD(b);   /*     [  m1  ]         */
    m2 = HI_WORD(a) * LO_WORD(b);   /*     [  m2  ]         */
    hi = HI_WORD(a) * HI_WORD(b);   /* [  hi  ]             */

    /* add m1 and m2 (middles) and carry into HI_WORD(hi) as needed */
    if (m2 && m1 >= (1 + ~m2))
        hi += (1L << 16);
    m1 += m2;

    /* seperate halves of (middles) */
    m2 = m1 >> 16;        /* HI_WORD(middles) aligned with hi */
    m1 = m1 << 16;        /* LO_WORD(middles) aligned with lo */

    /* add HI_WORD(middles) to hi */
    hi += m2;

    /* add LO_WORD(middles) to lo ... carry into hi as needed */
    if (m1 && lo >= (1 + ~m1))
        hi++;
    lo += m1;

    /* to effect rounding, add (1 << (n-1)) to lo ... carry into hi as needed */
    m1 = 1L << (n - 1);
    if (m1 && lo >= (1 + ~m1))
        hi++;
    lo += m1;

    /* overflow */
    if (hi >> n)
        return neg ? -MYINFINITY : MYINFINITY;

    /* [hi,lo] >> n */
    m1 = hi << (32 - n);
    m2 = lo >> n;

    hi = m1 | m2;
    if (hi > MYINFINITY)
        hi = MYINFINITY;
    return neg ? -(FS_LONG)hi : (FS_LONG)hi;

}
/****************************************************************/
FS_FIXED vardiv(FS_FIXED x, FS_FIXED y, int n)
{
    int neg;
    FS_FIXED q;
    FS_ULONG w, d;   /* q = w/d */
    /* FS_USHORT qct; */
    FS_SHORT qct;    /* 02-19-01 */

    if (y == 0)
        return (x >= 0) ? MYINFINITY : -MYINFINITY;
    if (x == 0)
        return 0L;

    neg = (x ^ y) < 0;
    if (x < 0) x = -x;
    if (y < 0) y = -y;

    w = x;
    d = y;
    qct = n + 1;  /* ABS(n)<32 */
    while (w < d)
    {
        w <<= 1;
        qct--;
    }
    while (d <= w)
    {
        d <<= 1;
        qct++;
    }
    d >>= 1;
    qct--;

    /* 02-19-01 */
    if (qct < 0)    /* <w> was too small with respect to <d> ... result is 0 */
        return 0;
    /* 02-19-01 */

    if (qct > 31)
        return neg ? -MYINFINITY : MYINFINITY;

    q = 0;
    while (qct)
    {
        q <<= 1;
        if (d <= w)      /* w==x, d==y */
        {
            w -= d;
            q |= 1;
        }
        w <<= 1;
        qct--;
    }

    if (d <= w)
        q++;

    return neg ? -q : q;
}
/****************************************************************/
#endif /* !defined(HAS_ASM) && !defined(HAS_FS_INT64) */


#if defined(HAS_FS_INT64) && !defined(HAS_ASM)
/****************************************************************/
/*************** using 64 bit integers **************************/
/****************************************************************/
/* note we do all of these unsigned to get the rounding correct */

FS_LONG muldiv_64(FS_LONG a, FS_LONG b, FS_LONG c)
{
    FS_INT64 x;
    FS_LONG neg;

    if (a == 0 || b == 0)
        return 0;
    neg = (a ^ b ^ c) < 0;
    if (c == 0)
        return neg ? -MYINFINITY : MYINFINITY;

    if (a < 0) a = -a;
    if (b < 0) b = -b;
    if (c < 0) c = -c;

    /* ? can we use a signed 32 bit int ? */
    /* a*b < 2^31 && c/2 < 0x7FFFFFFF - a*b <= 0x7FFFFFFF - 7FFEA810 = 88047 */
    if (a < 46340 && b < 46340 && c < 176096)
    {
        a *= b;
        a += (c >> 1);
        a /= c;
        return neg ? -a : a;
    }

    x = a;
    x *= b;
    x += (c >> 1);
    x /= c;

    if (x > MYINFINITY)
        x = MYINFINITY;

    return (FS_LONG) (neg ? -x : x);
}
/****************************************************************/
FS_LONG varmul_64(FS_FIXED a, FS_FIXED b, int n)
{
    FS_INT64 x;
    FS_LONG neg;

    if (a == 0 || b == 0)
        return 0;

    neg = (a ^ b) < 0;
    if (a < 0) a = -a;
    if (b < 0) b = -b;

    /* ? can we use a signed 32 bit int */
    if (a < 46340 && b < 46340)
    {
        a *= b;
        a += (1 << (n - 1));
        a >>= n;
        return neg ? -a : a;
    }

    x = a;
    x *= b;
    x += (FS_INT64)(1 << (n - 1));
    x >>= n;

    if (x > MYINFINITY)
        x = MYINFINITY;

    return (FS_LONG) (neg ? -x : x);
}
/****************************************************************/
FS_LONG vardiv_64(FS_LONG a, FS_LONG b, int n)
{
    FS_INT64 x;
    FS_LONG neg;

    if (b == 0)
        return (a >= 0) ? MYINFINITY : -MYINFINITY;
    if (a == 0)
        return 0;

    neg = (a ^ b) < 0;
    if (a < 0) a = -a;
    if (b < 0) b = -b;

    /* ? can we use a signed 32 bit int */
    /* we know (b>>1) < 2^30 */
    /* if (a << n) < 2^30 ... then (a<<n) + (b>>1) < 2^31 */
    /* (a << n) < (1<<30) <==>  a < (1 << (30 - n)) */
    if (a < (1 << (30 - n)))
    {
        a <<= n;
        a += (b >> 1);
        a /= b;
        return neg ? -a : a;
    }

    x = a;
    x <<= n;
    x += (b >> 1);
    x /= b;

    if (x > MYINFINITY)
        x = MYINFINITY;

    return (FS_LONG) (neg ? -x : x);
}

/****************************************************************/
FS_LONG vardiv_64NoRound(FS_LONG a, FS_LONG b, int n)
{
    FS_INT64 x;
    FS_LONG neg;

    if (b == 0)
        return (a >= 0) ? MYINFINITY : -MYINFINITY;
    if (a == 0)
        return 0;

    neg = (a ^ b) < 0;
    if (a < 0) a = -a;
    if (b < 0) b = -b;

    /* ? can we use a signed 32 bit int */
    /* we know (b>>1) < 2^30 */
    /* if (a << n) < 2^30 ... then (a<<n) + (b>>1) < 2^31 */
    /* (a << n) < (1<<30) <==>  a < (1 << (30 - n)) */
    if (a < (1 << (30 - n)))
    {
        a <<= n;
        a /= b;
        return neg ? -a : a;
    }

    x = a;
    x <<= n;
    x /= b;

    if (x > MYINFINITY)
        x = MYINFINITY;

    return (FS_LONG) (neg ? -x : x);
}

/****************************************************************/
#endif /* defined(HAS_FS_INT64) && !defined(HAS_ASM) */


#if defined(_MSC_VER) && defined(_M_IX86)

/*lint -e715 -e533*/

/****************************************************************/
/******* INTEL ASSEMBLY under MSVC ******************************/
/****************************************************************/
FS_LONG muldiv_asm(FS_LONG a,FS_LONG b,FS_LONG c)
{
    _asm {
        ;;; edx = ABS(b)           ;
        xor        eax,eax         ; initial result = 0;
        mov     edx,b              ;
        mov     ecx,edx            ; accumulate sign;
        or      edx,edx            ;
        jz        done             ; if (b==0) done;
        jg        b_pos            ;
        neg     edx                ;
b_pos:                             ;
        ;;; eax = ABS(a)           ;
        mov     eax,a              ;
        xor     ecx,eax            ; accumulate sign;
        or      eax,eax            ;
        jz        done             ;if (a==0) done;
        jg        a_pos            ;
        neg     eax                ;
a_pos:                             ;
        ;;; ebx = ABS(c)           ;
        mov     ebx,c              ;
        xor     ecx,ebx            ; accumulate sign;
        or      ebx,ebx            ;
        jz        bad              ;if (c==0) return +/- MYINFINITY;
        jg      c_pos              ;
        neg     ebx                ;
c_pos:                             ;
        ;;; ALL64 = a*b            ;
        mul     edx                ;
        ;;; ALL64 += c/2           ;
        push    ebx                ;
        sar     ebx,1              ;
        add     eax,ebx            ;
        adc     edx,0              ;
        pop        ebx             ;
        ;; if HI32 > c ... overflow;
        cmp     edx,ebx            ;
        jnb     bad                ;
        ;;; ALL64 / c              ;
        div     ebx                ;
        ;;; if MSB of quotient set ... overflow;
        or      eax,eax            ;
        jns        sign            ;
bad:                               ;
        ;;; result = MYINFINITY    ;
        mov        eax,7FFFFFFFh   ;
sign:                              ;
        ;;; correct sign of result ;
        or        ecx,ecx          ;
        jns        done            ;
        neg        eax             ;
done:                              ;
        }
}
/****************************************************************/
FS_FIXED varmul_asm(FS_FIXED a, FS_FIXED b,int n)
{
    _asm
        {
        ;;; ebx = ABS(b)             ;
        xor        eax,eax;          ; initial result=0;
        mov        ebx,b             ;
        mov        ecx,ebx           ; accumulate sign;
        or        ebx,ebx            ;
        jz        done               ; if (b==0) done;
        jg         b_pos             ;
        neg        ebx               ;
b_pos:                               ;
        ;;; eax = ABS(a)             ;
        mov        eax,a             ;
        xor        ecx,eax           ; accumulate sign;
        push    ecx;                 ; save accumulated sign;
        or        eax,eax            ;
        jz        sign               ; if (a==0) done;
        jg         a_pos             ;
        neg        eax               ;
a_pos:                               ;
        ;;; ALL64 == a*b             ;
        mul        ebx               ;
        ;;; temp = (1 << (n-1))      ;
        mov        ecx,n             ;
        dec        ecx               ;
        xor        ebx,ebx           ;
        inc        ebx               ;
        sal        ebx,cl            ;
        inc        ecx               ;
        ;;; ALL64 += temp            ;
        add        eax,ebx           ;
        adc        edx,0             ;
        ;;; ALL64 >>= n              ;
        shrd    eax,edx,cl           ;
        shr        edx,cl            ;
        ;;; if MSB set ... overflow  ;
        or        eax,eax            ;
        js        bad                ;
        ;;; HI32 nonzero ... overflow;
        or        edx,edx            ;
        jz        sign               ;
bad:                                 ;
        ;;; result = MYINFINITY      ;
        mov        eax,7FFFFFFFh     ;
sign:                                ;
        ;;; fix sign of result       ;
        pop        ecx               ;
        or        ecx,ecx            ;
        jns        done              ;
        neg        eax               ;
done:                                ;
        }
}
/****************************************************************/
FS_LONG vardiv_asm(FS_LONG x, FS_LONG y, int n)
{
    _asm {
        ;;; push sign of result    ;
        mov        ebx,y           ;
        mov        eax,x           ;
        mov        ecx,ebx         ;
        xor        ecx,eax         ;
        push    ecx                ;
        ;;; ebx := ABS(y)          ;
        or        ebx,ebx          ;
        jz        bad              ; if (y==0) return +/-MYINFINITY;
        jg        y_pos            ;
        neg        ebx             ;
y_pos:    ;;; eax := ABS(x)        ;
        or        eax,eax          ;
        jz        sign             ; if (x==0) return 0;
        jg        x_pos            ;
        neg        eax             ;
x_pos:    ;;; ALL64 <<= n          ;
        xor        edx,edx         ;
        mov        ecx,n           ;
        shld    edx,eax,cl         ;
        shl        eax,cl          ;
        ;;; ALL64 += y/2           ;
        push    ebx                ;
        sar     ebx,1              ;
        add     eax,ebx            ;
        adc     edx,0              ;
        pop        ebx             ;
        ;; if HI32 > y ... overflow;
        cmp     edx,ebx            ;
        jnb     bad                ;
        ;;; ALL64 / y              ;
        div     ebx                ;
        ;;; if MSB of quotient set ... overflow;
        or      eax,eax            ;
        jns        sign            ;
bad:    ;;; result = MYINFINITY    ;
        mov        eax,7FFFFFFFh   ;
sign:    ;;; correct sign          ;
        pop        ecx             ;
        or        ecx,ecx          ;
        jns        done            ;
        neg        eax             ;
done:                              ;
        }
}

/*lint +e715 +e533*/

/****************************************************************/
#endif /* defined(_WIN32) && defined(_M_IX86) */


/****************************************************************/
/**************** stand alone test programs *********************/
/****************************************************************/
#if EITHER_TEST
#include <time.h>
FS_VOID value(int);
FS_VOID timing(int);
static int seed;

/****************************************************************/
/* get the same pseudo random numbers on any platform */
int p_rand()
{
    return seed = ((16807L * seed + 127) & 0xFFFF);
}

/****************************************************************/
main(int argc, char *argv[])
{
    int n;

    /* get the random seed */
    if (2 == argc)
        n = atoi(argv[1]);
    else
        n = clock();

#if VALUE_TEST
    value(n);
#endif

#if TIMING_TEST
    timing(n);
#endif
}

#endif /* EITHER_TEST */

/****************************************************************/
#if VALUE_TEST
FS_VOID value(int arg)
{
    int i, n, m, a, b, c;
    double z, z6, z16, z30, sqrt();

    FS_PRINTF(("value : seed =%d\n", arg));
    seed = arg;
    z16 = (double)(1 << 16);
    z30 = (double)(1 << 30);
    z6 = (double)(1 << 6);

    /* do some tests */
    for (i = 0; i < 100; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = (p_rand() << 8) | p_rand();

        FS_PRINTF(("\na=%d b=%d c=%d\n", a, b, c));

        /*** MULDIV ***/
        n = muldiv(a, b, c);
        z = (a / z16) * (b / z16) / (c / z16);
        FS_PRINTF(("muldiv(%.5f,%.5f,%.5f)     =  %.5f [%.5f] \n", a / z16, b / z16, c / z16,  n / z16, z ));

#ifdef HAS_FS_INT64
        m = muldiv_64(a, b, c);
        if (m != n)
            FS_PRINTF(("***muldiv_64(%.5f,%.5f,%.5f)  = %.5f [%.5f] \n", a / z16, b / z16, c / z16, m / z16, z ));
#endif

#ifdef HAS_ASM
        m = muldiv_asm(a, b, c);
        if (m != n)
            FS_PRINTF(("**muldiv_asm(%.5f,%.5f,%.5f) = %.5f [%.5f] \n", a / z16, b / z16, c / z16, m / z16, z ));
#endif

        /*** FIXMUL ***/
        n = varmul(a, b, 16);
        z = (a / z16) * (b / z16);
        FS_PRINTF(("varmul(%.5f,%.5f,16)     = %.5f [%.5f]\n", a / z16, b / z16, n / z16, z));

#ifdef HAS_FS_INT64
        m = varmul_64(a, b, 16);
        if (m != n)
            FS_PRINTF(("***varmul_64(%.5f,%.5f,16)  = %.5f [%.5f]\n", a / z16, b / z16, m / z16, z));
#endif

#ifdef HAS_ASM
        m = varmul_asm(a, b, 16);
        if (m != n)
            FS_PRINTF(("***varmul_asm(%.5f,%.5f,16) = %.5f [%.5f]\n", a / z16, b / z16, m / z16, z));
#endif


        /*** FIXDIV ***/
        n = vardiv(a, b, 16);
        z = (a / z16) / (b / z16);
        FS_PRINTF(("vardiv(%.5f,%.5f,16)        = %.5f [%.5f]\n", a / z16, b / z16, n / z16, z));

#ifdef HAS_FS_INT64
        m = vardiv_64(a, b, 16);
        if (m != n)
            FS_PRINTF(("***vardiv_64(%.5f,%.5f,16)     = %.5f [%.5f]\n", a / z16, b / z16, m / z16, z));
#endif

#ifdef HAS_ASM
        m = vardiv_asm(a, b, 16);
        if (m != n)
            FS_PRINTF(("***vardiv_asm(%.5f,%.5f,16)    = %.5f [%.5f]\n", a / z16, b / z16, m / z16, z));
#endif


        /*** Mul26Dot6 ***/
        n = varmul(a, b, 6);
        z = (a / z6) * (b / z6);
        FS_PRINTF(("varmul(%.3f,%.3f,6)      = %.3f [%.3f]\n", a / z6, b / z6, n / z6, z));

#ifdef HAS_FS_INT64
        m = varmul_64(a, b, 6);
        if (m != n)
            FS_PRINTF(("***varmul_64(%.3f,%.3f,6)   = %.3f [%.3f]\n", a / z6, b / z6, m / z6, z));
#endif

#ifdef HAS_ASM
        m = varmul_asm(a, b, 6);
        if (m != n)
            FS_PRINTF(("***varmul_asm(%.3f,%.3f,6)  = %.3f [%.3f]\n", a / z6, b / z6, m / z6, z));
#endif


        /*** Div26Dot6 ***/
        n = vardiv(a, b, 6);
        z = (a / z6) / (b / z6);
        FS_PRINTF(("vardiv(%.3f,%.3f,6)     = %.3f [%.3f]\n", a / z6, b / z6, n / z6, z));

#ifdef HAS_FS_INT64
        m = vardiv_64(a, b, 6);
        if (m != n)
            FS_PRINTF(("***vardiv_64(%.3f,%.3f,6)  = %.3f [%.3f]\n", a / z6, b / z6, m / z6, z));
#endif

#ifdef HAS_ASM
        m = vardiv_asm(a, b, 6);
        if (m != n)
            FS_PRINTF(("***vardiv_asm(%.3f,%.3f,6) = %.3f [%.3f]\n", a / z6, b / z6, m / z6, z));
#endif


        /*** FRACMUL ***/
        n = varmul(a, b, 30);
        z = (a / z30) * (b / z30);
        FS_PRINTF(("varmul(%.8f,%.8f,30)     = %.8f [%.8f]\n", a / z30, b / z30, n / z30, z));

#ifdef HAS_FS_INT64
        m = varmul_64(a, b, 30);
        if (m != n)
            FS_PRINTF(("***varmul_64(%.8f,%.8f,30)  = %.8f [%.8f]\n", a / z30, b / z30, m / z30, z));
#endif

#ifdef HAS_ASM
        m = varmul_asm(a, b, 30);
        if (m != n)
            FS_PRINTF(("***varmul_asm(%.8f,%.8f,30) = %.8f [%.8f]\n", a / z30, b / z30, m / z30, z));
#endif


        /*** FRACDIV ***/
        n = vardiv(a, b, 30);
        z = (a / z30) / (b / z30);
        FS_PRINTF(("vardiv(%.8f,%.8f,30)     = %.8f [%.8f]\n", a / z30, b / z30, n / z30, z));

#ifdef HAS_FS_INT64
        m = vardiv_64(a, b, 30);
        if (m != n)
            FS_PRINTF(("***vardiv_64(%.8f,%.8f,30)  = %.8f [%.8f]\n", a / z30, b / z30, m / z30, z));
#endif

#ifdef HAS_ASM
        m = vardiv_asm(a, b, 30);
        if (m != n)
            FS_PRINTF(("***vardiv_asm(%.8f,%.8f,30) = %.8f [%.8f]\n", a / z30, b / z30, m / z30, z));
#endif


    }
}
#endif /* VALUE_TEST */

/****************************************************************/
#if TIMING_TEST
#define ITER 10000000L
FS_VOID timing(int arg)
{
    FS_LONG start, i, a, b, c, d;

    FS_PRINTF(("timing : seed =%d\n", arg));

    /* ANSI version */
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = (p_rand() << 8) | p_rand();
        d = muldiv(a, b, c);
    }
    FS_PRINTF(("muldiv          = %d\n", clock() - start));

#ifdef HAS_FS_INT64
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = (p_rand() << 8) | p_rand();
        d = muldiv_64(a, b, c);
    }
    FS_PRINTF(("muldiv_64       = %d\n", clock() - start));
#endif

#ifdef HAS_ASM
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = (p_rand() << 8) | p_rand();
        d = muldiv_asm(a, b, c);
    }
    FS_PRINTF(("muldiv_asm      = %d\n", clock() - start));
#endif

    /*** FixMul ***/
    /* ANSI version */
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = varmul(a, b, 16);
    }
    FS_PRINTF(("varmul          = %d\n", clock() - start));

#ifdef HAS_FS_INT64
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = varmul_64(a, b, 16);
    }
    FS_PRINTF(("varmul_64       = %d\n", clock() - start));
#endif

#ifdef HAS_ASM
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = varmul_asm(a, b, 16);
    }
    FS_PRINTF(("varmul_asm      = %d\n", clock() - start));
#endif


    /*** FixDiv */
    /* ANSI version */
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = vardiv(a, b, 16);
    }
    FS_PRINTF(("vardiv          = %d\n", clock() - start));

#ifdef HAS_FS_INT64
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = vardiv_64(a, b, 16);
    }
    FS_PRINTF(("vardiv_64       = %d\n", clock() - start));
#endif

#ifdef HAS_ASM
    seed = arg;
    start = clock();
    for (i = 0; i < ITER; i++)
    {
        a = (p_rand() << 12) | p_rand();
        b = (p_rand() << 8) | p_rand();
        c = vardiv_asm(a, b, 16);
    }
    FS_PRINTF(("vardiv_asm      = %d\n", clock() - start));
#endif


}
#endif /* TIMING_TEST */
/****************************************************************/
